<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    WC_MondialRelay
 * @subpackage WC_MondialRelay/includes
 * @author     Clément Barbaza <contact@mondialrelay-woocommerce.com>
 */
class WC_MondialRelay_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

    /**
     * The current version of WooCommerce.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $current_version    The current version of this plugin.
     */
    private $current_version;

    /**
     * The position of the plugin menu.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $menu_position    The menu position of this plugin.
     */
    private $menu_position;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version           The version of this plugin.
     * @param      string    $current_version   The current version of this plugin.
     * @param      string    $menu_position     The menu position of this plugin.
	 */
	public function __construct($plugin_name, $version, $current_version, $menu_position) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->current_version = $current_version;
		$this->menu_position = $menu_position;
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {
        wp_enqueue_style( 'woocommerce_admin', plugins_url().'/' . WC_MondialRelay::WOOCOMMERCE_FOLDER . '/assets/css/admin.css', $this->version, 'all' );
        wp_enqueue_style( 'mondialrelay_admin', plugin_dir_url(dirname(__FILE__)).'admin/css/woocommerce-mondialrelay-admin.css', $this->version, 'all' );
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {
        // Wordpress scripts
        wp_register_script( 'jquery', plugin_dir_url('https://ajax.googleapis.com/ajax/libs/jquery/1.11.2/jquery.min.js'));
        wp_enqueue_script('jquery', $this->version, false);
        // WooCommerce scripts
        $this->woocommerce_scripts();
        // Mondial Relay scripts
        wp_register_script( 'jquery_validate', plugin_dir_url(dirname(__FILE__)).'admin/js/jquery.validate.min.js', array( 'jquery'));
        wp_enqueue_script('jquery_validate', $this->version, false);
        wp_register_script( 'mondialrelay_admin_validate', plugin_dir_url(dirname(__FILE__)).'admin/js/woocommerce-mondialrelay-admin.js', array( 'jquery'));
        wp_enqueue_script('mondialrelay_admin_validate', $this->version, false);
	}

    /**
     * Register the WooCommerce JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function woocommerce_scripts() {
        // Register scripts
        wp_register_script( 'woocommerce_admin', plugins_url().'/' . WC_MondialRelay::WOOCOMMERCE_FOLDER . '/assets/js/admin/woocommerce_admin.min.js', array( 'jquery', 'jquery-blockui', 'jquery-ui-sortable', 'jquery-ui-widget', 'jquery-ui-core', 'jquery-tiptip' ), $this->version );
        wp_register_script( 'jquery-blockui', plugins_url().'/' . WC_MondialRelay::WOOCOMMERCE_FOLDER . '/assets/js/jquery-blockui/jquery.blockUI.min.js', array( 'jquery' ), '2.70', true );
        wp_register_script( 'jquery-tiptip', plugins_url().'/' . WC_MondialRelay::WOOCOMMERCE_FOLDER . '/assets/js/jquery-tiptip/jquery.tipTip.min.js', array( 'jquery' ), $this->version, true );
        // Enqueue scripts
        wp_enqueue_script( 'iris' );
        wp_enqueue_script( 'woocommerce_admin' );
    }

    /**
     * Show row meta on the plugin screen.
     *
     * @param	mixed $links Plugin Row Meta
     * @param	mixed $file  Plugin Base file
     * @return	array
     */
    public static function plugin_row_meta( $links, $file ) {
        if ( strpos( $file, WC_MondialRelay::PLUGIN_NAME ) !== false ) {
            $new_links = array(
                '<a href="' . WC_MondialRelay::DOCUMENTATION . '" target="_blank">' . __('Documentation', 'woocommerce_mondialrelay') . '</a>',
                '<a href="' . WC_MondialRelay::CLIENT . '" target="_blank">' . __('Espace client', 'woocommerce_mondialrelay') . '</a>',
            );
            $links = array_merge( $links, $new_links );
        }
        return $links;
    }

	/**
	 * Add a Mondial Relay menu
	 */
	public function MondialRelayMenu() {
		$mondialrelay_menu_position = get_option( 'mondialrelay_menu_position', WC_MondialRelay::MENU_POSITION );
		add_menu_page(
			__('Mondial Relay - Mondial Relay pour WooCommerce', 'woocommerce_mondialrelay'),
			__('Mondial Relay', 'woocommerce_mondialrelay'),
			'manage_woocommerce',
			'wc-mondialrelay',
			array($this, 'MondialRelayPage'),
			'dashicons-location',
			$mondialrelay_menu_position
		);
        add_submenu_page(
            'wc-mondialrelay',
            __('Réglages - Mondial Relay pour WooCommerce', 'woocommerce_mondialrelay'),
            __('Réglages', 'woocommerce_mondialrelay'),
            'manage_woocommerce',
            'wc-mondialrelay-configuration',
            array($this, 'MondialRelayConfiguration')
        );
		add_submenu_page(
			'wc-mondialrelay',
			__('Expéditions - Mondial Relay pour WooCommerce', 'woocommerce_mondialrelay'),
			__('Expéditions', 'woocommerce_mondialrelay'),
			'manage_woocommerce',
			'wc-mondialrelay-expeditions',
            array($this, 'MondialRelayShipping')
		);
		add_submenu_page(
			'wc-mondialrelay',
			__('Maintenance - Mondial Relay pour WooCommerce', 'woocommerce_mondialrelay'),
			__('Maintenance', 'woocommerce_mondialrelay'),
			'manage_woocommerce',
			'wc-mondialrelay-maintenance',
            array($this, 'MondialRelayMaintenance')
		);

    }

	/**
	 * Notification when the plugin has an update available
	 */
	public function notificationMenu() {
		if (false !== ($contents = @file_get_contents($this->current_version))) {
			$current = file_get_contents($this->current_version);
		} else {
			$current = $this->version;
		}
		if (version_compare($this->version, $current, '<')) {
			$submenu['wc-mondialrelay'][5][0] .= "<span class='update-plugins count-1'><span class='update-count'>1</span></span>";
			$menu[$this->menu_position][0] .= "<span class='update-plugins count-1'><span class='update-count'>1</span></span>";
		}
	}

    /**
     * Mondial Relay options page
     * @param object $settings
     */
    function MondialRelayPage($settings) {
        global $wpdb;
        $prefixe = $wpdb->prefix;

        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/woocommerce-mondialrelay-admin-page.php';
    }

    /**
     * Mondial Relay config page
     * @param object $settings
     */
    function MondialRelayConfiguration($settings) {
        global $wpdb;
        $prefixe = $wpdb->prefix;

        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/woocommerce-mondialrelay-admin-configuration.php';
    }

    /**
     * Expeditions listing page of Mondial Relay
     * @param object $settings
     */
    function MondialRelayShipping($settings) {
        global $wpdb;
        $prefixe = $wpdb->prefix;

        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/woocommerce-mondialrelay-admin-shipping.php';
    }

    /**
     * Mondial Relay maintenance page
     * @param object $settings
     */
    function MondialRelayMaintenance($settings) {
        global $wpdb;
        $prefixe = $wpdb->prefix;

        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/woocommerce-mondialrelay-admin-maintenance.php';
    }

    /**
     * Mondial Relay in a WooCommerce order
     * @param  int $order_id
     */
    function actions_mr_meta_box($order_id) {
        add_meta_box(
            'mondialrelay_expedition_meta_box',
            __( 'Mondial Relay', 'actions_mondialrelay' ),
            array($this, 'actions_mr_meta_box_callback'),
            'shop_order',
            'side'
        );
    }

    /**
     * Mondial Relay function in a WooCommerce order
     * @param  object $order
     */
    function actions_mr_meta_box_callback( $order ) {
        global $wpdb;
        $prefixe = $wpdb->prefix;
        $order_id = $order->ID;

        // Check if the order uses Mondial Relay
        $is_mondial_relay = WC_MondialRelay_Sql::get_meta_champ($order_id, 'ID Point Mondial Relay', 1);
        if (isset($is_mondial_relay) && $is_mondial_relay == 1) {
            // Display the right box
            wp_nonce_field( 'actions_mr_meta_box', 'actions_mr_meta_box_nonce' );
            // Check if a Mondial Relay expedition already exists
            $exist_expedition = WC_MondialRelay_Sql::get_meta_champ($order_id, 'ExpeditionNum', 1);
            // A Mondial Relay expedition already exists for this order
            if (isset($exist_expedition) && $exist_expedition) {
                $expedition_num = WC_MondialRelay_Sql::get_meta_champ($order_id, 'ExpeditionNum');
                $point_mondial_relay = WC_MondialRelay_Sql::get_meta_champ($order_id, 'ID Point Mondial Relay');
                ?>
                <table class="wp-list-table widefat fixed striped">
                    <tbody>
                    <tr>
                        <td width="60%" style="text-align: left"><?php _e('Expédition', 'woocommerce_mondialrelay') ?></td>
                        <td width="40%" style="text-align: right"><strong><?php echo $expedition_num ?></strong></td>
                    </tr>
                    <tr>
                        <td width="60%" style="text-align: left"><?php _e('Point Mondial Relay', 'woocommerce_mondialrelay') ?></td>
                        <td width="40%" style="text-align: right"><strong><?php echo $point_mondial_relay ?></strong></td>
                    </tr>
                    </tbody>
                </table>
                <?php
                // Get the global parameters of Mondial Relay account
                $params_compte = WC_MondialRelay_Order::get_params_compte();
                // Get the Mondial Relay point
                $id_mondial_relay = WC_MondialRelay_Sql::get_meta_champ($order_id, 'ID Point Mondial Relay');
                $explode_id_mondial_relay = explode('-', $id_mondial_relay);
                $liv_rel_pays_mondial_relay = $explode_id_mondial_relay[0];
                $liv_rel_mondial_relay = $explode_id_mondial_relay[1];
                // SOAP
                $client = new soapclient(WC_MondialRelay::MONDIALRELAY_WEBSERVICE);
                $params = array(
                    'Enseigne'       => $params_compte['mondialrelay_code_client'],
                    'Expeditions'    => $expedition_num,
                    'Langue'         => $liv_rel_pays_mondial_relay,
                );
                // Generation of the security key
                $code = implode("", $params);
                $code .= $params_compte['mondialrelay_cle_privee'];
                $params["Security"] = strtoupper(md5($code));
                $etiquette = $client->WSI2_GetEtiquettes($params)->WSI2_GetEtiquettesResult;
                if ($etiquette->STAT == 0) {
                    // Colis tracking
                    add_thickbox();
                    // Format 10x15
                    $etiquette->URL_PDF_10x15 = str_replace('A4', '10x15', $etiquette->URL_PDF_A4);
                    ?>
                    <p style="text-align:center">
                        <a class="button button-default" target="_blank" href="http://www.mondialrelay.fr/<?php echo $etiquette->URL_PDF_A4 ?>">
                            <?php _e('Étiquette A4', 'woocommerce_mondialrelay') ?>
                        </a>
                        <a class="button button-default" target="_blank" href="http://www.mondialrelay.fr/<?php echo $etiquette->URL_PDF_A5 ?>">
                            <?php _e('Étiquette A5', 'woocommerce_mondialrelay') ?>
                        </a>
                    </p>
                    <p style="text-align:center">
                        <a class="button button-default" target="_blank" href="http://www.mondialrelay.fr/<?php echo $etiquette->URL_PDF_10x15 ?>">
                            <?php _e('Étiquette 10x15', 'woocommerce_mondialrelay') ?>
                        </a>
                    </p>
                    <p style="text-align:center">
                        <a href="#TB_inline?width=600&height=420&inlineId=mondial_relay_tracking_thickbox" class="button button-default thickbox mondialrelay-tracking-button" title="<?php _e("Suivi d'expédition", 'woocommerce-mondialrelay') ?>" data-order="<?php echo $order_id ?>" data-expedition="<?php echo $expedition_num ?>" data-langue="FR">
                            <?php _e("Suivi", 'woocommerce-mondialrelay') ?>
                        </a>
                    </p>
                    <?php require_once dirname(__FILE__) . '/partials/woocommerce-mondialrelay-admin-tracking-view.php'; ?>
                    <?php
                }
                else { ?>
                    <p style="color: rgb(255,84,96);">
                        <?php _e('Erreur', 'woocommerce_mondialrelay')?> <?php echo $etiquette->STAT ?>
                    </p>
                <?php }
            }
            // No Mondial Relay expedition for this order
            else {
                // Get the default parameters
                $mondialrelay_code_colis = get_option( 'mondialrelay_mode_colis', 1 );
                $mondialrelay_code_livraison = get_option( 'mondialrelay_mode_livraison', 1 );
                // Get the weight of the order
                $order = new WC_Order( $order_id );
                $items = $order->get_items();
                $poids_array = array();
                foreach ( $items as $item ) {
                    $product_id = $item['product_id'];
                    $poids = get_post_meta($product_id, '_weight');
                    if (isset($poids)) {
                        $poids = get_post_meta($product_id, '_weight',true);
                    } else {
                        $poids = "0";
                    }
                    $poids = $poids * $item['qty'];
                    // Convert the weight in g according to the WooCoomerce weight unit selected
                    $poids = WC_MondialRelay_Helpers::convert_poids($poids);
                    $poids_array[] = $poids;
                }
                $poids = array_sum($poids_array);
                // View
                add_thickbox();
                $data = array(
                    'ModeCol' => $mondialrelay_code_colis,
                    'ModeLiv' => $mondialrelay_code_livraison,
                    'Poids' => $poids,
                );
                $this->actions_mr_meta_box_create_view($order_id, $data);
            }
        }
        else { ?>
            <em>
                <?php _e("Cette commande n'utilise pas Mondial Relay", "woocommerce_mondialrelay") ?>
            </em>
        <?php
        }
    }

    /**
     * Mondial Relay creation of expedition view
     *
     * @param $order_id
     * @param $data
     */
    function actions_mr_meta_box_create_view($order_id, $data) {
        ?>
        <div style="text-align: center">
                    <a href="#TB_inline?width=600&height=420&inlineId=mondial_relay_expedition_thickbox" class="button button-default thickbox" title="<?php _e('Créer une expédition Mondial Relay', 'woocommerce_mondialrelay') ?>"><?php _e("Créer une expédition", "woocommerce_mondialrelay") ?></a>
        </div>
        <div id="mondial_relay_expedition_thickbox" style="display:none;">
            <div id="mr-notif" style="font-size: 13px; display:none"></div>
            <div id="mr-text" style="display: none; text-align: center; margin-top: 20px;">Veuillez patienter pendant le rechargement de la page...</div>
            <div id="mr-form">
            <h3><?php _e('Paramètres', 'woocommerce_mondialrelay') ?></h3>
            <table class="form-table">
                <tbody>
                <tr valign="top">
                    <th scope="row" class="titledesc">
                        <label for="ModeCol"><?php _e('Mode de collecte', 'woocommerce_mondialrelay') ?></label>
                        <span class="woocommerce-help-tip" data-tip="<?php _e('Le mode de collecte de colis', 'woocommerce_mondialrelay') ?>"></span>
                    </th>
                    <td>
                        <select name="ModeCol" id="ModeCol" required="required">
                            <option value="" disabled><?php _e('Choisir', 'woocommerce_mondialrelay') ?></option>
                            <option class="level-0" value="CCC" <?php if ($data['ModeCol'] == 'CCC') echo 'selected'; ?>>CCC</option>
                            <option class="level-0" value="CDR" <?php if ($data['ModeCol'] == 'CDR') echo 'selected'; ?>>CDR</option>
                            <option class="level-0" value="CDC" <?php if ($data['ModeCol'] == 'CDC') echo 'selected'; ?>>CDC</option>
                            <option class="level-0" value="REL" <?php if ($data['ModeCol'] == 'REL') echo 'selected'; ?>>REL</option>
                        </select>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row" class="titledesc">
                        <label for="ModeLiv"><?php _e('Mode de livraison', 'woocommerce_mondialrelay') ?></label>
                        <span class="woocommerce-help-tip" data-tip="<?php _e('Le mode de livraison de colis', 'woocommerce_mondialrelay') ?>"></span>
                    </th>
                    <td>
                        <select name="ModeLiv" id="ModeLiv" required="required">
                            <option value="" disabled><?php _e('Choisir', 'woocommerce_mondialrelay') ?></option>
                            <option class="level-0" value="LCC" <?php if ($data['ModeLiv'] == 'LCC') echo 'selected'; ?>>LCC</option>
                            <option class="level-0" value="LD1" <?php if ($data['ModeLiv'] == 'LD1') echo 'selected'; ?>>LD1</option>
                            <option class="level-0" value="LDS" <?php if ($data['ModeLiv'] == 'LDS') echo 'selected'; ?>>LDS</option>
                            <option class="level-0" value="24R" <?php if ($data['ModeLiv'] == '24R') echo 'selected'; ?>>24R</option>
                            <option class="level-0" value="24L" <?php if ($data['ModeLiv'] == '24L') echo 'selected'; ?>>24L</option>
                            <option class="level-0" value="24X" <?php if ($data['ModeLiv'] == '24X') echo 'selected'; ?>>24X</option>
                            <option class="level-0" value="ESP" <?php if ($data['ModeLiv'] == 'ESP') echo 'selected'; ?>>ESP</option>
                            <option class="level-0" value="DRI" <?php if ($data['ModeLiv'] == 'DRI') echo 'selected'; ?>>DRI</option>
                        </select>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row" class="titledesc">
                        <label for="Poids"><?php _e('Poids', 'woocommerce_mondialrelay') ?></label>
                        <span class="woocommerce-help-tip" data-tip="<?php _e('Selon l\'unité définie dans WooCommerce (grammes par défaut)', 'woocommerce_mondialrelay') ?>"></span>
                    </th>
                    <td>
                        <input name="Poids" id="Poids" type="number" value="<?php echo $data['Poids'] ?>" placeholder="<?php _e('Obligatoire', 'woocommerce_mondialrelay') ?>" maxlength="7" required="required">
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row" class="titledesc">
                        <label for="NbColis"><?php _e('Nombre de colis', 'woocommerce_mondialrelay') ?></label>
                        <span class="woocommerce-help-tip" data-tip="<?php _e('Nombre de colis de l\'expédition', 'woocommerce_mondialrelay') ?>"></span>
                    </th>
                    <td>
                        <input type="number" value="1" name="NbColis" id="NbColis"  placeholder="<?php _e('Obligatoire', 'woocommerce_mondialrelay') ?>"  maxlength="2" required="required">
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row" class="titledesc">
                        <label for="NbColis"><?php _e('Contre-remboursement', 'woocommerce_mondialrelay') ?></label>
                        <span class="woocommerce-help-tip" data-tip="<?php _e('Contre-remboursement (en centimes)', 'woocommerce_mondialrelay') ?>"></span>
                    </th>
                    <td>
                        <input type="number" value="0" name="CRT_Valeur" id="CRT_Valeur" placeholder="<?php _e('Obligatoire', 'woocommerce_mondialrelay') ?>" maxlength="7" required="required">
                    </td>
                </tr>
                </tbody>
            </table>
            <div style="text-align: right">
                <a href="#" id="options-expedition-link"><?php _e('Paramètres avancés', 'woocommerce_mondialrelay'); ?></a>
            </div>
            <div id="options-expedition" style="display: none">
                <h3><?php _e('Paramètres avancés', 'woocommerce_mondialrelay') ?></h3>
                <table class="form-table">
                    <tbody>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="NDossier"><?php _e('Numéro de dossier', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e('Information libre', 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <input type="text" value="<?php echo $order_id ?>" name="NDossier" id="NDossier" placeholder="<?php _e('Facultatif', 'woocommerce_mondialrelay') ?>" maxlength="15">
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="NClient"><?php _e('Numéro de client', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e('Information libre', 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <input type="text" name="NClient" id="NClient" placeholder="<?php _e('Facultatif', 'woocommerce_mondialrelay') ?>"  maxlength="9">
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="Longueur"><?php _e('Longueur', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e('Longueur développée en cm', 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <input type="number" name="Longueur" id="Longueur" placeholder="<?php _e('Facultatif', 'woocommerce_mondialrelay') ?>"  maxlength="3">
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="Taille"><?php _e('Taille', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e('Taille (sauf indication contraire de la part de Mondial Relay, veuillez ne rien renseigner dans cette zone)', 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <select name="Taille" id="Taille">
                                    <option value=""><?php _e('Choisir', 'woocommerce_mondialrelay') ?></option>
                                    <option class="level-0" value="XS">XS</option>
                                    <option class="level-0" value="S">S</option>
                                    <option class="level-0" value="M">M</option>
                                    <option class="level-0" value="L">L</option>
                                    <option class="level-0" value="XL">XL</option>
                                    <option class="level-0" value="XXL">XXL</option>
                                    <option class="level-0" value="3XL">3XL</option>
                                </select>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="CRT_Devise"><?php _e('Devise de contre-remboursement', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e('Devise de contre-remboursement', 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <select name="CRT_Devise" id="CRT_Devise">
                                    <option value=""><?php _e('Choisir', 'woocommerce_mondialrelay') ?></option>
                                    <option class="level-0" value="EUR">Euro</option>
                                </select>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="Exp_Valeur"><?php _e('Valeur de l\'expédition', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e('Valeur de l\'expédition, en centimes dans la devise', 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <input type="number" name="Exp_Valeur" id="Exp_Valeur" placeholder="<?php _e('Facultatif', 'woocommerce_mondialrelay') ?>"  maxlength="7">
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="Exp_Devise"><?php _e('Devise de l\'expédition', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e('Devise de l\'expédition', 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <select name="Exp_Devise" id="Exp_Devise">
                                    <option value=""><?php _e('Choisir', 'woocommerce_mondialrelay') ?></option>
                                    <option class="level-0" value="EUR">Euro</option>
                                </select>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="TAvisage"><?php _e('Demande d\'avisage', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e("Demande d'avisage pour cette expédition. Attention : les règles d'avisage Mondial Relay peuvent outrepasser une demande explicite. Veuillez contacter Mondial Relay afin de savoir le fonctionnement normal des avisages. Si vous n'êtes pas concerné par les avisages, vous pouvez systématiquement laisser cette zone vide.", 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <select name="TAvisage" id="TAvisage">
                                    <option value=""><?php _e('Choisir', 'woocommerce_mondialrelay') ?></option>
                                    <option class="level-0" value="O">Oui</option>
                                    <option class="level-0" value="N">Non</option>
                                </select>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="TReprise"><?php _e('Demande de reprise', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e("Demande de reprise'", 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <select name="TReprise" id="TReprise">
                                    <option value=""><?php _e('Choisir', 'woocommerce_mondialrelay') ?></option>
                                    <option class="level-0" value="O">Oui</option>
                                    <option class="level-0" value="N">Non</option>
                                </select>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="Montage"><?php _e('Montage', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e('Temps de montage', 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <input type="number" name="Montage" id="Montage" placeholder="<?php _e('Facultatif', 'woocommerce_mondialrelay') ?>"  maxlength="3">
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="TRDV"><?php _e('Rendez-vous', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e("Demande de rendez-vous. Attention : les règles de Mondial Relay prévalent sur une demande explicite. Veuillez ne pas utiliser ce paramètres et le laisser systématiquement vide.", 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <select name="TRDV" id="TRDV">
                                    <option value=""><?php _e('Choisir', 'woocommerce_mondialrelay') ?></option>
                                    <option class="level-0" value="O">Oui</option>
                                    <option class="level-0" value="N">Non</option>
                                </select>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row" class="titledesc">
                                <label for="Instructions"><?php _e('Instructions', 'woocommerce_mondialrelay') ?></label>
                                <span class="woocommerce-help-tip" data-tip="<?php _e('Instructions complémentaires', 'woocommerce_mondialrelay') ?>"></span>
                            </th>
                            <td>
                                <textarea name="Instructions" id="Instructions" cols="31" rows="1" maxlength="31" style="resize: none" placeholder="<?php _e('Facultatif', 'woocommerce_mondialrelay') ?>"></textarea>
                            </td>
                        </tr>
                    </tbody>
                    </table>
            </div>
            <hr>
            <div style="text-align:center">
                <a href="#" id="create-expedition" class="button save_order button-primary" data-order-id="<?php echo $order_id ?>">
                    <?php _e('Créer une expédition', 'woocommerce_mondialrelay') ?>
                </a>
            </div>
            </div>
        </div>
        <?php
    }

    /**
     * Javascript script for the creation of the Mondial Relay expedition
     */
    function expedition_javascript() { ?>
        <script type="text/javascript">
            jQuery(document).ready(function($) {

                jQuery('#options-expedition-link').on('click', function(e) {
                    e.preventDefault();
                    jQuery('#options-expedition').toggle();
                });

                jQuery('#create-expedition').on('click', function(e) {
                    e.preventDefault();
                    var th = jQuery(this);
                    var data = {
                        'action': 'expedition',
                        'order': th.attr('data-order-id'),
                        'ModeCol': jQuery('#ModeCol').val(),
                        'ModeLiv': jQuery('#ModeLiv').val(),
                        'Poids': jQuery('#Poids').val(),
                        'NbColis': jQuery('#NbColis').val(),
                        'CRT_Valeur': jQuery('#CRT_Valeur').val(),
                        'NDossier': jQuery('#NDossier').val(),
                        'NClient': jQuery('#NClient').val(),
                        'Longueur': jQuery('#Longueur').val(),
                        'Taille': jQuery('#Taille').val(),
                        'CRT_Devise': jQuery('#CRT_Devise').val(),
                        'Exp_Valeur': jQuery('#Exp_Valeur').val(),
                        'Exp_Devise': jQuery('#Exp_Devise').val(),
                        'TAvisage': jQuery('#TAvisage').val(),
                        'TReprise': jQuery('#TReprise').val(),
                        'Montage': jQuery('#Montage').val(),
                        'TRDV': jQuery('#TRDV').val(),
                        'Instructions': jQuery('#Instructions').val(),
                    };

                    $.post(ajaxurl, data, function(response) {
                        if (response != 0) {
                            var expedition = <?php echo __("'L\'expédition n\'a pas pu être créée'", 'woocommerce_mondialrelay') ?>;
                            var erreur = <?php echo __("'Erreur'", 'woocommerce_mondialrelay') ?>;
                            jQuery('#mr-notif').addClass('update-nag').css({ "margin": "0", "display": "block" }).html("<strong>"+expedition+"</strong><br>"+erreur+" "+response).show();
                        }
                        else {
                            var expedition = <?php echo __("'L\'expédition a été créée'", 'woocommerce_mondialrelay') ?>;
                            jQuery('#mr-notif').addClass('updated').css({ "padding": "11px 15px" }).html("<strong>"+expedition+"</strong>").show();
                            jQuery('#mr-form').hide();
                            jQuery('#mr-text').show();
                            window.location.reload();
                        }
                        console.log(response);
                    });
                });
            });
        </script>
    <?php }

    /**
     * Expedition creation in a Mondial Relay order (AJAX)
     */
    function expedition_callback() {
        global $wpdb;
        $prefixe = $wpdb->prefix;

        // Get the debug mode
        $mondialrelay_debug = get_option('mondialrelay_debug', WC_MondialRelay::DEBUG);

        // Get the order number
        $order_id = intval( $_POST['order'] );

        // Get the Mondial Relay parameters
        $params_compte = WC_MondialRelay_Order::get_params_compte();

        // Mondial Relay WebService
        $client = new soapclient(WC_MondialRelay::MONDIALRELAY_WEBSERVICE);

        // Get the Mondial Relay point
        $id_mondial_relay = WC_MondialRelay_Sql::get_meta_champ($order_id, 'ID Point Mondial Relay');
        $explode_id_mondial_relay = explode('-', $id_mondial_relay);
        $params_livraison = WC_MondialRelay_Order::get_params_livraison();
        $params_livraison['liv_rel_pays_mondial_relay'] = $explode_id_mondial_relay[0];
        $params_livraison['liv_rel_mondial_relay'] = $explode_id_mondial_relay[1];

        // Get the sender parameters
        $params_expediteur = WC_MondialRelay_Order::get_params_expediteur();

        // Get the recipient parameters
        $params_destinataire = WC_MondialRelay_Order::get_params_destinataire($order_id);

        // Get the order parameters
        $params_commande = WC_MondialRelay_Order::get_params_commande();

        // Get assurance
        $assurance = get_option('mondialrelay_assurance', WC_MondialRelay::ASSURANCE);

        // Get the collect point parameters
        $params_collecte = WC_MondialRelay_Order::get_params_collecte();
        // Mode REL
        if ($params_commande['ModeCol'] == 'REL') {
            $params_collecte['col_rel_pays_mondial_relay'] = $params_livraison['liv_rel_pays_mondial_relay'];
            $params_collecte['col_rel_mondial_relay'] = $params_livraison['liv_rel_mondial_relay'];
        }

        // Expedition parameters
        $params = array(
            'Enseigne'        => $params_compte['mondialrelay_code_client']
            ,'ModeCol'        => $params_commande['ModeCol']
            ,'ModeLiv'        => $params_commande['ModeLiv']
            ,'NDossier'       => $params_commande['NDossier']
            ,'NClient'        => $params_commande['NClient']
            ,'Expe_Langage'   => $params_expediteur['mondialrelay_vendeur_langage']
            ,'Expe_Ad1'       => $params_expediteur['mondialrelay_vendeur_adresse1']
            ,'Expe_Ad2'       => $params_expediteur['mondialrelay_vendeur_adresse2']
            ,'Expe_Ad3'       => $params_expediteur['mondialrelay_vendeur_adresse3']
            ,'Expe_Ad4'       => $params_expediteur['mondialrelay_vendeur_adresse4']
            ,'Expe_Ville'     => $params_expediteur['mondialrelay_vendeur_ville']
            ,'Expe_CP'        => $params_expediteur['mondialrelay_vendeur_cp']
            ,'Expe_Pays'      => $params_expediteur['mondialrelay_vendeur_pays']
            ,'Expe_Tel1'      => $params_expediteur['mondialrelay_vendeur_tel']
            ,'Expe_Tel2'      => $params_expediteur['mondialrelay_vendeur_tel2']
            ,'Expe_Mail'      => $params_expediteur['mondialrelay_vendeur_email']
            ,'Dest_Langage'   => $params_destinataire['_shipping_langage']
            ,'Dest_Ad1'       => $params_destinataire['_billing_full_name']
            ,'Dest_Ad2'       => $params_destinataire['_billing_company']
            ,'Dest_Ad3'       => $params_destinataire['_shipping_address_1']
            ,'Dest_Ad4'       => $params_destinataire['_shipping_address_2']
            ,'Dest_Ville'     => $params_destinataire['_shipping_city']
            ,'Dest_CP'        => $params_destinataire['_shipping_postcode']
            ,'Dest_Pays'      => $params_destinataire['_shipping_country']
            ,'Dest_Tel1'      => $params_destinataire['_billing_phone']
            ,'Dest_Mail'      => $params_destinataire['_billing_email']
            ,'Poids'          => $params_commande['Poids']
            ,'Longueur'       => $params_commande['Longueur']
            ,'Taille'         => $params_commande['Taille']
            ,'NbColis'        => $params_commande['NbColis']
            ,'CRT_Valeur'     => $params_commande['CRT_Valeur']
            ,'CRT_Devise'     => $params_commande['CRT_Devise']
            ,'Exp_Valeur'     => $params_commande['Exp_Valeur']
            ,'Exp_Devise'     => $params_commande['Exp_Devise']
            ,'COL_Rel_Pays'	  => $params_collecte['col_rel_pays_mondial_relay']
            ,'COL_Rel'	 	  => $params_collecte['col_rel_mondial_relay']
            ,'LIV_Rel_Pays'   => $params_livraison['liv_rel_pays_mondial_relay']
            ,'LIV_Rel'        => $params_livraison['liv_rel_mondial_relay']
            ,'TAvisage'       => $params_commande['TAvisage']
            ,'TReprise'       => $params_commande['TReprise']
            ,'Montage'        => $params_commande['Montage']
            ,'TRDV'           => $params_commande['TRDV']
            ,'Instructions'   => $params_commande['Instructions']
            ,'Assurance'      => $assurance
        );

        // Generation of the security key
        $code = implode("", $params);
        $code .= $params_compte['mondialrelay_cle_privee'];
        $params["Security"] = strtoupper(md5($code));

        // Expedition creation
        $expedition = $client->WSI2_CreationExpedition($params)->WSI2_CreationExpeditionResult;
        $callback = $expedition->STAT;
        // Statut code
        if ($callback != 0) {
            echo $callback . '<br>' . WC_MondialRelay_Helpers::statut($callback);
            if ($mondialrelay_debug) {
                WC_MondialRelay_Helpers::VarPrint($params);
            }
        }
        elseif ($callback == 0) {
            // Update the checkout status
            $checkout_status = get_option( 'mondialrelay_state_order', '' );
            if ($checkout_status) {
                $wpdb->update($prefixe . 'posts', array('post_status' => $checkout_status), array('ID' => $order_id));
            }
            // Add the Mondial Relay expedition paramaters in the database
            $champs_bdd = WC_MondialRelay_Order::get_expedition_champs($expedition, $params_commande['NbColis']);
            WC_MondialRelay_Sql::insert_meta_champs($order_id, $champs_bdd);

            // Email
            // Get the Mondial Relay email
            $email = WC_MondialRelay_Order::get_params_emails($order_id);
            // br in html
            $email['message'] = nl2br($email['message']);
            // Check if email send is activated
            if ($email['send'] == 1) {
                $destinataire = $params_destinataire['_billing_email'];
                // Send the Mondial Relay tracking email to the client
                $mail = wc_mail($destinataire, $email['sujet'], $email['message']);
            }
        }

        die();
    }

    /**
     * Javascript script for the tracking of the Mondial Relay expedition
     */
    function tracking_javascript() { ?>
        <script type="text/javascript">
            jQuery(document).ready(function($) {

                jQuery('.mondialrelay-tracking-button').on('click', function(e) {

                    var th = jQuery(this);
                    var order = th.attr('data-order');
                    var expedition = th.attr('data-expedition');
                    var langue = th.attr('data-langue');
                    var data = {
                        'action': 'tracking',
                        'order': order,
                        'expedition': expedition,
                        'langue': langue
                    };

                    $.post(ajaxurl, data, function(response) {
                        var obj = JSON.parse(response);
                        //console.log(obj);
                        var jsonObj = JSON.parse(obj.content);
                        var jsonPretty = JSON.stringify(jsonObj, null, '\t');
                        if (obj.response) {
                            jQuery('#mondial_relay_tracking_notif').addClass('updated').css({ "padding": "11px 15px" }).html(obj.notification).show();

                            jQuery('#mondial_relay_tracking_Libelle01').html('<strong>' + obj.content.Libelle01 + '</strong>');
                            jQuery('#mondial_relay_tracking_Relais_Libelle').html('<strong>' + obj.content.Relais_Libelle + '</strong>');
                            jQuery('#mondial_relay_tracking_Relais_Num').html('<strong>' + obj.content.Relais_Num + '</strong>');
                            jQuery('#mondial_relay_tracking_Libelle02').html('<strong>' + obj.content.Libelle02 + '</strong>');

                            var tracing_view = '<table class="wp-list-table widefat fixed striped"><tbody>';
                            for (var i = 0; i < tracing.length; i++){
                                var table = tracing[i];
                                tracing_view += '<tr><td colspan="2"><strong>Évènement ' + i + '</strong></td></tr>';
                                for (var key in table) {
                                    var tracing_libelle = key.replace('Tracing_', '');
                                    tracing_view += '<tr>';
                                    tracing_view += '<td width="60%" style="text-align: left">' + tracing_libelle + '</td>';
                                    tracing_view += '<td width="40%" style="text-align: right"><strong>' + table[key] + '</strong></td>';
                                    tracing_view += '</tr>';
                                }
                            }
                            tracing_view += '</tbody></table>';
                            //console.log(tracing_view);

                            jQuery('#mondial_relay_tracking_tracing').html(tracing_view);
                            jQuery('#mondial_relay_tracking_content').show();

                        }
                        else {
                            jQuery('#mondial_relay_tracking_notif').addClass('update-nag').css({ "display": "block" }).html(obj.notification).show();
                        }
                        if (obj.debug) {
                            jQuery('#mondial_relay_tracking_debug').html(jsonPretty);
                        }
                        //console.log(response);

                    });

                });

            });
        </script>
    <?php }

    /**
     * Expedition tracking of a Mondial Relay expedition (AJAX)
     */
    function tracking_callback() {
        // Get the debug mode
        $mondialrelay_debug = get_option('mondialrelay_debug', WC_MondialRelay::DEBUG);
        // Get the Mondial Relay parameters
        $params_compte = WC_MondialRelay_Order::get_params_compte();
        // Get the shipping parameters
        $expedition = $_POST['expedition'];
        $langue = $_POST['langue'];
        // SOAP
        $client = new soapclient(WC_MondialRelay::MONDIALRELAY_WEBSERVICE);
        $params = array(
            'Enseigne'       => $params_compte['mondialrelay_code_client'],
            'Expeditions'    => $expedition,
            'Langue'         => $langue,
        );
        // Generation of the security key
        $code = implode("", $params);
        $code .= $params_compte['mondialrelay_cle_privee'];
        $params["Security"] = strtoupper(md5($code));
        $tracking = $client->WSI2_TracingColisDetaille($params)->WSI2_TracingColisDetailleResult;
        if ($tracking->STAT == 0) {
            $response = 1;
            $notification  = 'Succès';
        }
        else {
            $response = 0;
            $notification  = $tracking->STAT  . ' : ' . WC_MondialRelay_Helpers::statut($tracking->STAT );
        }
        $debug = 0;
        if ($mondialrelay_debug) {
            $debug = 1;
        }
        $data = [
            'response' => $response,
            'notification' => $notification,
            'content' => json_encode( $tracking ),
            'debug' => $debug,
        ];
        echo json_encode( $data );
        die();
    }

}
