<?php
/**
 * Systempay V2-Payment Module version 1.2.3 (revision 66856) for WooCommerce 2.x.
 *
 * Copyright (C) 2014-2015 Lyra Network and contributors
 * Support contact : supportvad@lyra-network.com
 * Author link : http://www.lyra-network.com/
 * Contributors : Alsacréations (Geoffrey Crofte http://alsacreations.fr/a-propos#geoffrey)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * @category  payment
 * @package   systempay
 * @author    Lyra Network <supportvad@lyra-network.com>
 * @copyright 2014-2015 Lyra Network and contributors
 * @license   http://www.gnu.org/licenses/old-licenses/gpl-2.0.html  GNU General Public License (GPL v2)
 * @version   1.2.3 (revision 66856)
*/

/**
 * Systempay Payment Gateway : standard payment class.
 */
class WC_Gateway_SystempayStd extends WC_Gateway_Systempay {

	public function __construct() {
		$this->id = 'systempaystd';
		$this->icon = apply_filters('woocommerce_systempaystd_icon', WC_SYSTEMPAY_PLUGIN_URL . '/assets/images/systempay.png');
		$this->has_fields = false;
		$this->method_title = 'Systempay - ' . __('One-time Payment', 'systempay');

		// init Systempay common vars
		$this->systempay_init();

		// load the form fields
		$this->init_form_fields();

		// load the module settings
		$this->init_settings();

		// define user set variables
		$this->title = $this->settings['title'];
		$this->description = $this->settings['description'];
		$this->testmode = ($this->settings['ctx_mode'] == 'TEST');
		$this->debug = (isset($this->settings['debug']) && $this->settings['debug'] == 'yes') ? true : false;

		// reset Systempay standard payment admin form action
		add_action( 'woocommerce_settings_start', array($this, 'systempay_reset_admin_options'));

		// update Systempay standard payment admin form action
		add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));

		// generate Systempay standard payment form action
		add_action('woocommerce_receipt_' . $this->id, array($this, 'systempay_generate_form'));

		// return from payment platform action
		add_action('woocommerce_api_wc_gateway_systempay', array($this, 'systempay_notify_response'));
	}

	/**
	 * Get icon function.
	 *
	 * @access public
	 * @return string
	 */
	public function get_icon() {
		global $woocommerce;

		$icon = '';

		if($this->icon) {
			$icon = '<img style="width: 85px;" src="';
			$icon .= class_exists('WC_HTTPS') ? WC_HTTPS::force_https_url($this->icon) : $woocommerce->force_ssl($this->icon);
			$icon .= '" alt="' . $this->title . '" />';
		}

		return apply_filters('woocommerce_gateway_icon', $icon, $this->id );
	}

	/**
	 * Initialise Gateway Settings Form Fields.
	 */
	public function init_form_fields() {
		global $woocommerce;

		// load common form fields to concat them with sub-module settings
		parent::init_form_fields();
		foreach ($this->form_fields as $k => $v) {
			$this->systempay_common_fields[$k] = $v;
		}

		$this->form_fields = array(
				// CMS config params
				'module_settings' => array(
						'title' => __('MODULE SETTINGS', 'systempay'),
						'type' => 'title'
				),
				'enabled' => array(
						'title' => __('Activation', 'systempay'),
						'label' => __('Enable / disable', 'systempay'),
						'type' => 'checkbox',
						'default' => 'yes',
						'description' => __('Enables / disables standard payment.', 'systempay')
				),
				'title' => array(
						'title' => __('Title', 'systempay'),
						'type' => 'text',
						'description' => __('This controls the title which the user sees during checkout.', 'systempay'),
						'default' => 'Paiement par carte bancaire'
				),
				'description' => array(
						'title' => __( 'Description', 'systempay' ),
						'type' => 'textarea',
						'description' => __( 'This controls the description which the user sees during checkout.', 'systempay' ),
						'default' => 'Vous allez être redirigé(e) vers la page de paiement après confirmation de la commande.',
				),

				// amount restrictions
				'amount_restrictions' => array(
						'title' => __('AMOUNT RESTRICTIONS', 'systempay'),
						'type' => 'title'
				),
				'amount_min' => array(
						'title' => __('Minimum amount', 'systempay'),
						'type' => 'text',
						'default' => '',
						'description' => __('Minimum amount to activate this payment method.', 'systempay')
				),
				'amount_max' => array(
						'title' => __('Maximum amount', 'systempay'),
						'type' => 'text',
						'default' => '',
						'description' => __( 'Maximum amount to activate this payment method.', 'systempay')
				),
		);
	}

	/**
	 * Override init_settings methode to retrieve Systempay common settings.
	 */
	public function init_settings() {
		parent::init_settings();

		$common_settings = get_option('woocommerce_systempay_settings', null );

		// if there are no settings defined, load defaults
		if ((!$common_settings || !is_array($common_settings)) && isset($this->systempay_common_fields) && is_array($this->systempay_common_fields)) {
			foreach ($this->systempay_common_fields as $k => $v) {
				$this->settings[$k] = isset($v['default']) ? $v['default'] : '';
			}
		} else {
			foreach ($common_settings as $k => $v) {
				$this->settings[$k] =  $v ;
			}
		}
	}

	/**
	 * Check if this gateway is enabled and available for the current cart.
	 */
	public function is_available() {
		global $woocommerce;

		if(!$this->is_supported_currency()) {
			return false;
		}

		$amount = $woocommerce->cart->total;
		if (($this->settings['amount_max'] != '' && $amount > $this->settings['amount_max'])
				|| ($this->settings['amount_min'] != '' && $amount < $this->settings['amount_min'])) {
			return false;
		}

		return parent::is_available();
	}


	/**
	 * Process the payment and return the result.
	 **/
	public function process_payment($order_id) {
		global $woocommerce;

		$order = new WC_Order($order_id);

		if(version_compare($woocommerce->version, '2.1.0', '<')) {
			$pay_url = add_query_arg('order', $order->id, add_query_arg('key', $order->order_key, get_permalink(woocommerce_get_page_id('pay'))));
		} else {
			$pay_url = $order->get_checkout_payment_url(true);
		}

		return array(
				'result' 	=> 'success',
				'redirect'	=> $pay_url
		);
	}

	/**
	 * Order review and payment form page.
	 **/
	public function systempay_generate_form($order_id) {
		global $woocommerce;

		$order = new WC_Order($order_id);

		echo '<div style="opacity: 0.6; padding: 10px; text-align: center; color: #555;	border: 3px solid #aaa; background-color: #fff; cursor: wait; line-height: 32px;">';

		$img_url = WC_SYSTEMPAY_PLUGIN_URL . 'assets/images/loading.gif';
		$img_url = class_exists('WC_HTTPS') ? WC_HTTPS::force_https_url($img_url) : $woocommerce->force_ssl($img_url);
		echo '<img src="' . esc_url( $img_url ) . '" alt="..." style="float:left; margin-right: 10px;"/>';
		echo __('Please wait, you will be redirected to the payment platform.', 'systempay');
		echo '</div>';
		echo '<br />';
		echo '<p>'.__('If nothing happens in 10 seconds, please click the button below.', 'systempay').'</p>';

		$this->systempay_fill_request($order);

		$form = "\n".'<form action="' . esc_url($this->systempay_request->get('platform_url')) . '" method="post" name="' . $this->id . '_payment_form" target="_top">';
		$form .= "\n".$this->systempay_request->getRequestHtmlFields();
		$form .= "\n".'<input type="submit" class="button-alt" id="submit_' . $this->id . '_payment_form" value="' . sprintf(__('Pay via %s', 'systempay'), 'Systempay').'"/>';
		$form .= "\n".'<a class="button cancel" href="' . esc_url($order->get_cancel_order_url() ) . '">'.__('Cancel order &amp; restore cart', 'systempay') . '</a>';
		$form .= "\n".'</form>';

		$form .= "\n".'<script type="text/javascript">';
		$form .= "\nfunction systempay_submit_form() {
				    document.getElementById('submit_" . $this->id . "_payment_form').click();
				  }";
		$form .= "\nif (window.addEventListener) { // for all major browsers, except IE 8 and earlier
				    window.addEventListener('load', systempay_submit_form, false);
				  } else if (window.attachEvent) { // for IE 8 and earlier versions
				    window.attachEvent('onload', systempay_submit_form);
				  }";
		$form .= "\n</script>\n";

		echo $form;
	}

	/**
	 * Prepare Systempay form params to send to payment gateway.
	 **/
	protected function systempay_fill_request($order) {
		global $woocommerce, $wpdb;

		$this->log('Generating payment form for order #' . $order->id . '.');

		// get currency
		$currency = SystempayApi::findCurrencyByAlphaCode(get_woocommerce_currency());
		if($currency == null) {
			$this->log('The store currency (' . get_woocommerce_currency() . ') is not supported by Systempay.');

			wp_die(sprintf(__('The store currency (%s) is not supported by %s.'), get_woocommerce_currency(), 'Systempay'));
		}

		// effective used version
		include ABSPATH . WPINC . '/version.php';
		$version = $wp_version . '-' . $woocommerce->version;

		// Systempay params
		$misc_params = array(
				'amount' => $currency->convertAmountToInteger($order->get_total()),
				'contrib' => 'WooCommerce2.x_1.2.3/' . $version,
				'currency' => $currency->getNum(),
				'order_id' => $order->id,
				'order_info' => $order->order_key,
				'order_info2' => 'blog_id=' . $wpdb->blogid, // save blog_id for multisite cases

				// billing address info
				'cust_id' => $order->user_id,
				'cust_email' => $order->billing_email,
				'cust_first_name' => $order->billing_first_name,
				'cust_last_name' => $order->billing_last_name,
				'cust_address' => $order->billing_address_1 . ' ' .  $order->billing_address_2,
				'cust_zip' => $order->billing_postcode,
				'cust_country' => $order->billing_country,
				'cust_phone' => str_replace(array('(', '-', ' ', ')'), '', $order->billing_phone),
				'cust_city' => $order->billing_city,
				'cust_state' => $order->billing_state,

				// shipping address info
				'ship_to_first_name' => $order->shipping_first_name,
				'ship_to_last_name' => $order->shipping_last_name,
				'ship_to_street' => $order->shipping_address_1,
				'ship_to_street2' => $order->shipping_address_2,
				'ship_to_city' => $order->shipping_city,
				'ship_to_state' => $order->shipping_state,
				'ship_to_country' => $order->shipping_country,
				'ship_to_zip' => $order->shipping_postcode,
				'ship_to_phone' => str_replace(array('(', '-', ' ', ')'), '', $order->shipping_phone),

				// return URLs
				'url_return' => add_query_arg('wc-api', 'WC_Gateway_Systempay', home_url('/'))
		);
		$this->systempay_request->setFromArray($misc_params);

		// activate 3ds ?
		$threeds_mpi = null;
		if($this->settings['3ds_min_amount'] != '' && $order->get_total() < $this->settings['3ds_min_amount']) {
			$threeds_mpi = '2';
		}
		$this->systempay_request->set('threeds_mpi', $threeds_mpi);

		// detect language
		$locale = get_locale() ? substr(get_locale(), 0, 2) : null;
		if($locale && SystempayApi::isSupportedLanguage($locale)) {
			$this->systempay_request->set('language', $locale);
		} else {
			$this->systempay_request->set('language', $this->settings['language']);
		}

		// available languages
		$langs = $this->settings['available_languages'];
		if(is_array($langs) && !in_array('', $langs)) {
			$this->systempay_request->set('available_languages', implode(';', $langs));
		}

		// payment cards
		$cards = $this->settings['payment_cards'];
		if(is_array($cards) && !in_array('', $cards)) {
			$this->systempay_request->set('payment_cards', implode(';', $cards));
		}

		// enable automatic redirection ?
		$this->systempay_request->set('redirect_enabled', ($this->settings['redirect_enabled'] == 'yes') ? true : false);

		// other configuration params
		$config_keys = array(
				'site_id', 'key_test', 'key_prod', 'ctx_mode', 'platform_url', 'capture_delay', 'validation_mode',
				'redirect_success_timeout', 'redirect_success_message', 'redirect_error_timeout',
				'redirect_error_message', 'return_mode'
		);

		foreach($config_keys as $key) {
			$this->systempay_request->set($key, $this->settings[$key]);
		}
	}
}